﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.EventSystems;

public class InputManager : MonoBehaviour
{
    private Dictionary<string, bool> _AxisDown; //A dictionary containing booleans for each axis to indicate whether it is down
    private static InputManager _Instance;

    public static InputManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<InputManager>();
            }

            return _Instance;
        }
    }

    #region EventArgs Classes
    public class InputStateChangedEventArgs : EventArgs
    {
        public Constants.InputState OldState { get; set; }  //The old input state
        public Constants.InputState NewState { get; set; }  //The new input state
        public bool Handled { get; set; } = false;  //Whether the event has already been handled
    }

    public event EventHandler<InputStateChangedEventArgs> InputStateChanged;
    #endregion

    public Constants.InputState InputState
    {
        get
        {
            return _InputState;
        }

        set
        {
            InputStateChangedEventArgs args = new InputStateChangedEventArgs();
            args.OldState = _InputState;
            args.NewState = value;
            _InputState = value;

            if (PageManager.CurrentPage != null)
            {
                PageManager.CurrentPage.ReFocus();
            }

            OnInputStateChanged(args);
        }
    }

    private Constants.InputState _InputState;

    private void Start()
    {
        //Default to mouse and keyboard, add the axis's to monitor
        InputState = Constants.InputState.MouseAndKeyboard;
        _AxisDown = new Dictionary<string, bool>();

        _AxisDown["Controller Left Stick X"] = false;
        _AxisDown["Controller Left Stick Y"] = false;
        _AxisDown["Controller Triggers"] = false;
        _AxisDown["Controller Right Stick X"] = false;
        _AxisDown["Controller Right Stick Y"] = false;
        _AxisDown["Controller D-Pad X"] = false;
        _AxisDown["Controller D-Pad Y"] = false;
    }

    void Update()
    {
        List<string> keys = new List<string>(_AxisDown.Keys);
        foreach(string axis in keys)
        {  
            //Loop through the axis's
            if (_AxisDown[axis])
            {
                //The axis was down, let's check if it's still down
                _AxisDown[axis] = !(Input.GetAxis(axis) == 0.0f);
            }
        }
    }

    private void OnGUI()
    {
        switch (InputState)
        {
            case Constants.InputState.Controller:
                if (isKeyboardInput())
                {
                    //We're using a controller but just detected keyboard input, time to switch
                    InputState = Constants.InputState.MouseAndKeyboard;
                    SwitchInputDevice();
                }
                break;

            case Constants.InputState.MouseAndKeyboard:
                if (isControllerInput())
                {
                    //We're using a keyboard but just detected controller input, time to switch
                    InputState = Constants.InputState.Controller;
                    SwitchInputDevice();
                }
                break;
        }
    }

    private bool isKeyboardInput()
    {
        if (Event.current.isKey || Event.current.isMouse)
        {
            //Key or mouse button input detected, we're using keyboard
            return true;
        }

        if (Input.GetAxis("Mouse X") != 0.0f || Input.GetAxis("Mouse Y") != 0.0f)
        {
            //Mouse movement detected, we're using keyboard
            return true;
        }

        return false;
    }

    private bool isControllerInput()
    {
        // joystick buttons
        if (Input.GetKey("joystick button 0") ||
           Input.GetKey("joystick button 1") ||
           Input.GetKey("joystick button 2") ||
           Input.GetKey("joystick button 3") ||
           Input.GetKey("joystick button 4") ||
           Input.GetKey("joystick button 5") ||
           Input.GetKey("joystick button 6") ||
           Input.GetKey("joystick button 7") ||
           Input.GetKey("joystick button 8") ||
           Input.GetKey("joystick button 9") ||
           Input.GetKey("joystick button 10") ||
           Input.GetKey("joystick button 11") ||
           Input.GetKey("joystick button 12") ||
           Input.GetKey("joystick button 13") ||
           Input.GetKey("joystick button 14") ||
           Input.GetKey("joystick button 15") ||
           Input.GetKey("joystick button 16") ||
           Input.GetKey("joystick button 17") ||
           Input.GetKey("joystick button 18") ||
           Input.GetKey("joystick button 19"))
        {
            //A joystick key was pressed, we're using controller
            return true;

        }

        if (Input.GetAxis("Controller Left Stick X") != 0.0f ||
           Input.GetAxis("Controller Left Stick Y") != 0.0f ||
           Input.GetAxis("Controller Triggers") != 0.0f ||
           Input.GetAxis("Controller Right Stick X") != 0.0f ||
           Input.GetAxis("Controller Right Stick Y") != 0.0f ||
           Input.GetAxis("Controller D-Pad X") != 0.0f ||
           Input.GetAxis("Controller D-Pad Y") != 0.0f)
        {
            //A controller axis was moved, we're using controller
            return true;
        }

        return false;
    }

    /// <summary>
    /// Returns a value on the axis if the axis isn't already down i.e. it's unique
    /// </summary>
    /// <param name="axis"></param>
    /// <returns></returns>
    public float GetControllerAxisUnique(string axis)
    {
        if(!_AxisDown[axis] && Input.GetAxis(axis) != 0.0f)
        {
            //We weren't down but just detected movement, so we are now, let's return the value
            _AxisDown[axis] = true;
            return Input.GetAxis(axis);
        }

        return 0.0f;
    }

    private void SwitchInputDevice()
    {
        if (InputState == Constants.InputState.Controller)
        {
            //Hide and lock the cursor as we're using a controller
            Cursor.visible = false;
            Cursor.lockState = CursorLockMode.Locked;
        }

        else
        {
            //Unhide and free the cursor as we're using keyboard
            Cursor.visible = true;
            Cursor.lockState = CursorLockMode.None;
        }
    }

    private void OnInputStateChanged(InputStateChangedEventArgs e)
    {
        //Called when the input state changes and sends an event notification to all subscribers
        EventHandler<InputStateChangedEventArgs> handler = InputStateChanged;
        if (handler != null)
        {
            handler(this, e);
        }
    }
}
